﻿using Duende.IdentityModel.Client;
using Microsoft.IdentityModel.Tokens;
using System;
using System.Collections.Generic;
using System.IdentityModel.Tokens.Jwt;
using System.Net.Http;
using System.Security.Claims;
using System.Security.Cryptography.X509Certificates;
using System.Threading.Tasks;

namespace Viss.ApiManagement.Samples.Utils
{
    public class IdentityServerClient
    {
        private readonly HttpClient httpClient;
        private readonly ClientCredentialsTokenRequest clientCredentialstokenRequest;
        private readonly TokenIntrospectionRequest tokenIntrospectionRequest;

        public IdentityServerClient(ClientCredentialsTokenRequest clientCredentialstokenRequest, TokenIntrospectionRequest tokenIntrospectionRequest)
        {
            this.httpClient = new HttpClient();
            this.clientCredentialstokenRequest = clientCredentialstokenRequest;
            this.tokenIntrospectionRequest = tokenIntrospectionRequest;
        }

        public async Task<TokenResponse> GetAccessTokenClientCredentialsAsync()
        {
            var tokenResponse = await httpClient.RequestClientCredentialsTokenAsync(clientCredentialstokenRequest);

            if (tokenResponse.IsError)
            {
                Console.WriteLine("Token request failed:");
                Console.WriteLine($"Error: {tokenResponse.Error}");
                Console.WriteLine($"ErrorDescription: {tokenResponse.ErrorDescription}");
                Console.WriteLine($"HttpStatusCode: {tokenResponse.HttpStatusCode}");
                Console.WriteLine($"HttpErrorReason: {tokenResponse.HttpErrorReason}");
                Console.WriteLine($"Raw: {tokenResponse.Raw}");

                throw new Exception($"Failed getting system user token : {tokenResponse.Error ?? tokenResponse.ErrorDescription}");
            }

            return tokenResponse;
        }

        public async Task<TokenIntrospectionResponse> GetIntrospectionAsync(string token)
        {
            tokenIntrospectionRequest.Token = token;
            var introspectionResponse = await httpClient.IntrospectTokenAsync(tokenIntrospectionRequest);
            if (introspectionResponse.IsError)
            {
                throw new Exception($"Failed getting token introspect : {introspectionResponse.Error}");
            }
            return introspectionResponse;
        }

        internal static string CreateAssertion(X509Certificate2 clientCertificate, string clientId, string identityAddress, bool includeCertificate = true, bool includeThumbrint = false)        {
            var validFrom = DateTime.UtcNow;
            var validTo = validFrom + TimeSpan.FromSeconds(600);
            var jwtHeader = new JwtHeader(new X509SigningCredentials(clientCertificate));
            if (includeCertificate)
            {
                jwtHeader["x5c"] = Convert.ToBase64String(clientCertificate.GetRawCertData());
            }
            
            if (includeThumbrint)
            {
                jwtHeader["x5t"] = Convert.ToBase64String(clientCertificate.GetCertHash());
            }

            var jwtPayload = new JwtPayload(
                issuer: clientId,
                audience: identityAddress,
                claims: new List<Claim>() { 
                    new Claim("sub", clientId), 
                    new Claim("jti", Guid.NewGuid().ToString()), 
                },
                notBefore: validFrom,
                expires: validTo
                );
            var jwt = new JwtSecurityToken(jwtHeader, jwtPayload);
            var handler = new JwtSecurityTokenHandler();
            var assertion = handler.WriteToken(jwt);
            return assertion;
        }

    }
}
