﻿using Duende.IdentityModel.Client;
using Microsoft.Extensions.Caching.Memory;
using System;
using System.Net;
using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;

namespace Viss.ApiManagement.Samples.Utils
{
    public class AuthenticationHandler : DelegatingHandler
    {
        private const string BearerTokenKey = "BearerTokenKey";
        private readonly IdentityServerClient identityServerClient;
        private readonly IMemoryCache cache;
        private readonly TokenType tokenType;

        public AuthenticationHandler(IdentityServerClient identityServerClient, TokenType tokenType) : base(new HttpClientHandler())
        {
            this.identityServerClient = identityServerClient;
            this.cache = new MemoryCache(new MemoryCacheOptions());
            this.tokenType = tokenType;
        }

        protected override async Task<HttpResponseMessage> SendAsync(HttpRequestMessage request, CancellationToken cancellationToken)
        {
            HttpResponseMessage response = null;
            async Task retry()
            {
                var tokenResponse = await identityServerClient.GetAccessTokenClientCredentialsAsync(); //or await identityServerClient.GetAccessTokenClientCredentials()

                if (tokenResponse != null && !string.IsNullOrEmpty(tokenResponse.AccessToken))
                {

                    if (tokenType == TokenType.Jwt)
                    {
                        Console.WriteLine("Introspection not supported for JWT tokens");
                        Console.WriteLine("Access token:");
                        Console.WriteLine(tokenResponse.AccessToken);
                    }
                    else
                    {
                        var introspection = await identityServerClient.GetIntrospectionAsync(tokenResponse.AccessToken);

                        Console.WriteLine($"Introspection response for token {tokenResponse.AccessToken}");
                        Console.WriteLine(introspection.Raw);
                        Console.WriteLine();
                    }

                    cache.Set(BearerTokenKey, tokenResponse.AccessToken,
                        new MemoryCacheEntryOptions().SetAbsoluteExpiration(TimeSpan.FromSeconds(tokenResponse.ExpiresIn * 0.8)));
                    request.SetBearerToken(tokenResponse.AccessToken);
                    response = await base.SendAsync(request, cancellationToken);
                }
            }

            if (cache.TryGetValue(BearerTokenKey, out string cachedToken) && !string.IsNullOrEmpty(cachedToken))
            {
                request.SetBearerToken(cachedToken);
                response = await base.SendAsync(request, cancellationToken);
            }
            else
            {
                await retry();
            }

            if (response.StatusCode == HttpStatusCode.Unauthorized
                || response.StatusCode == HttpStatusCode.Forbidden)
            {
                await retry();
            }

            return response;
        }

    }
}
