﻿using CalculationReference;
using Duende.IdentityModel.Client;
using Microsoft.Win32.SafeHandles;
using System;
using System.Net.Http;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Threading.Tasks;
using System.Xml.Linq;
using Viss.ApiManagement.Samples.TransactionApi;
using Viss.ApiManagement.Samples.Utils;

namespace Viss.ApiManagement.Samples
{
    public class SoapExample
    {
        const string baseAddress = "https://apitestgw.vraa.gov.lv/ISS-VARAM_VRAA-PaulsCalcNoScopeWSDL/v1_0/";
        const string identityAddress = "https://viss-portal-test.vraa.gov.lv/IamIdentity.Server/connect/token";
        
        const string jwtClientId = "urn:oauth2:d4ed79fa-8aaf-42aa-86ab-7d3db8defbdc";
        const string referenceClientId = "urn:oauth2:bd9f5fc0-e527-4ab1-8c8a-b870433435f2";

        const string pathnameprivkeyfile = "TestCert.pfx";
        const string passwordprivkey = "123";
        const string scope = "urn:viss.apimanagement-apicall API-SIA.ABC-Viss.ApiManagement.TransactionApi-v1_0-IamStart API-SIA.ABC-Viss.ApiManagement.TransactionApi-v1_0-IamInfo";
        const string transactionApiAddress = "https://vissapi-test.vraa.gov.lv/ApiManagement.TransactionApi";
        const string eServiceId = "URN:IVIS:100001:EP-EP186-v1-0";

        const string introspectionAddress = "https://viss-portal-test.vraa.gov.lv/IamIdentity.Server/connect/introspect";
        const string introspectClientId = "urn:viss.apimanagement";
        const string introspectClientSecret = "secret";

        private readonly CalculationSyncClient client;
        private readonly IdentityServerClient identityServerClient;
        private readonly TransactionApiClient transactionApiClient;

        // Iespejams parslegties starp JWT un Reference tokeniem mainot tokenTypeToUse mainīgā vērtību
        // JWT tokeniem jadarbojās bez introspekcijas
        // Reference tokeniem ir introspekcija.
        private readonly TokenType tokenTypeToUse = TokenType.Jwt;

        private string transactionId;
        private string accessToken;

        public SoapExample()
        {
            var clientCertificate = new X509Certificate2(pathnameprivkeyfile, passwordprivkey);
            string effectiveClientId = tokenTypeToUse == TokenType.Jwt ? jwtClientId : referenceClientId;

            this.identityServerClient = new IdentityServerClient(
                new ClientCredentialsTokenRequest()
                {
                    Address = identityAddress,
                    ClientId = effectiveClientId,
                    Scope = scope,
                    ClientCredentialStyle = ClientCredentialStyle.PostBody,
                    ClientAssertion = new ClientAssertion()
                    {
                        Type = "urn:ietf:params:oauth:client-assertion-type:jwt-bearer",
                        Value = IdentityServerClient.CreateAssertion(
                            clientCertificate, 
                            effectiveClientId, 
                            identityAddress, 
                            includeCertificate: true, 
                            includeThumbrint: false),
                    }
                },
                new TokenIntrospectionRequest()
                {
                    Address = introspectionAddress,
                    ClientId = introspectClientId,
                    ClientSecret = introspectClientSecret
                }
            );

            this.transactionApiClient = new TransactionApiClient(new HttpClient()
            {
                BaseAddress = new Uri(transactionApiAddress)
            });
            
            this.client = new CalculationSyncClient(new BasicHttpBinding(BasicHttpSecurityMode.Transport), new EndpointAddress(baseAddress));
        }

        public async Task RunAsync()
        {
            await MultiplyAsync();
            await DivideAsync();
        }

        private async Task EnsureAccessTokenAsync()
        {
            if (accessToken != null)
            {
                return;
            }

            TokenResponse tokenResponse = await identityServerClient.GetAccessTokenClientCredentialsAsync();
            accessToken = tokenResponse?.AccessToken;

            if (string.IsNullOrEmpty(accessToken))
            {
                throw new Exception("Access token is null or empty");
            }

            if (tokenTypeToUse == TokenType.Jwt)
            {
                Console.WriteLine("Introspection not supported for JWT tokens");
                Console.WriteLine("Access token:");
                Console.WriteLine(accessToken);
            }
            else
            {
                var introspection = await identityServerClient.GetIntrospectionAsync(accessToken);
                Console.WriteLine($"Introspection response for token {accessToken}");
                Console.WriteLine(introspection.Raw);
                Console.WriteLine();
            }
        }

        private async Task EnsureTransactionAsync()
        {
            if (transactionId != null)
            {
                return;
            }

            transactionApiClient.HttpClient.SetBearerToken(accessToken);
            transactionId = await transactionApiClient.StartAsync(eServiceId);
        }

        private async Task MultiplyAsync()
        {
            await EnsureAccessTokenAsync();
            await EnsureTransactionAsync();

            using (new OperationContextScope(client.InnerChannel))
            {
                OperationContext.Current.OutgoingMessageProperties[HttpRequestMessageProperty.Name] = new HttpRequestMessageProperty()
                {
                    Headers =
                    {
                        { "Authorization", $"Bearer {accessToken}" },
                        { "x-transactionId", transactionId },
                    }
                };

                RequestHeaderStructure header = new RequestHeaderStructure()
                {
                    MessageID = "6cafe2c6-9834-47ad-b710-be8033710b02",
                    MessageType = "URN:IVIS:100001:XSD-Testing-TestISServise-v1-0-TYPE-Calculation",
                    TransactionID = transactionId,
                    CorrelationID = "f4ceae8a-dfd9-48b3-a0b6-5d990aaac86e",
                    Destination = "URN:IVIS:100001:ISS-SIA.ABC-CalculationDataSync-v1-0",
                    MilestoneID = "URN:IVIS:100001:EP-EdkTester-v1-0-MS-CallCalcSync",
                    Sender = new ParticipantStructure()
                    {
                        SystemID = "URN:IVIS:100001:PORTAL-Unknown",
                        Item = new InhabitantStructure()
                        {
                            PersonID = new PersonIDStructure()
                            {
                                Scheme = "urn:ivis:100001:name.id-viss",
                                Value = "PK:09016711212"
                            },
                            PersonCode = "09016711212",
                            FullName = new PersonFullNameStructure()
                            {
                                LastName = "Rožkalns",
                                FirstName = "Aivars"
                            }
                        }
                    }
                };

                IVISRequestStructure request = new IVISRequestStructure()
                {
                    Header = header,
                    Body = XElement.Parse(@"
                                    <Calculation xmlns=""http://ivis.eps.gov.lv/XMLSchemas/100000/TestISServise/v1-0"">
                                        <Number1>5</Number1>
	                                    <Number2>6</Number2>
	                                    <Operation>multiplication</Operation>
                                    </Calculation>")
                };

                DefaultMethodSyncResponse response = await client.DefaultMethodSyncAsync(request);
                Console.WriteLine(response.IVISResponse.Body.ToString());
            }
        }

        private async Task DivideAsync()
        {
            await EnsureAccessTokenAsync();
            await EnsureTransactionAsync();

            using (new OperationContextScope(client.InnerChannel))
            {
                OperationContext.Current.OutgoingMessageProperties[HttpRequestMessageProperty.Name] = new HttpRequestMessageProperty()
                {
                    Headers =
                    {
                        { "Authorization", $"Bearer {accessToken}" },
                        { "x-transactionId", transactionId },
                    }
                };

                RequestHeaderStructure header = new RequestHeaderStructure()
                {
                    MessageID = "6cafe2c6-9834-47ad-b710-be8033710b02",
                    MessageType = "URN:IVIS:100001:XSD-Testing-TestISServise-v1-0-TYPE-Calculation",
                    TransactionID = transactionId,
                    CorrelationID = "f4ceae8a-dfd9-48b3-a0b6-5d990aaac86e",
                    Destination = "URN:IVIS:100001:ISS-SIA.ABC-CalculationDataSync-v1-0",
                    MilestoneID = "URN:IVIS:100001:EP-EdkTester-v1-0-MS-CallCalcSync",
                    Sender = new ParticipantStructure()
                    {
                        SystemID = "URN:IVIS:100001:PORTAL-Unknown",
                        Item = new InhabitantStructure()
                        {
                            PersonID = new PersonIDStructure()
                            {
                                Scheme = "urn:ivis:100001:name.id-viss",
                                Value = "PK:09016711212"
                            },
                            PersonCode = "09016711212",
                            FullName = new PersonFullNameStructure()
                            {
                                LastName = "Rožkalns",
                                FirstName = "Aivars"
                            }
                        }
                    }
                };


                IVISRequestStructure request = new IVISRequestStructure()
                {
                    Header = header,
                    Body = XElement.Parse(@"
                                    <Calculation xmlns=""http://ivis.eps.gov.lv/XMLSchemas/100000/TestISServise/v1-0"">
                                        <Number1>10</Number1>
	                                    <Number2>2</Number2>
	                                    <Operation>division</Operation>
                                    </Calculation>")
                };

                DefaultMethodSyncResponse response = await client.DefaultMethodSyncAsync(request);
                Console.WriteLine(response.IVISResponse.Body.ToString());
            }
        }
    }
}
