﻿using Duende.IdentityModel.Client;
using System;
using System.Collections.Generic;
using System.Net.Http;
using System.Net.Http.Json;
using System.Security.Cryptography.X509Certificates;
using System.Threading.Tasks;
using Viss.ApiManagement.Samples.TransactionApi;
using Viss.ApiManagement.Samples.Utils;

namespace Viss.ApiManagement.Samples
{
    public class RestExample
    {
        const string baseAddress = "https://apitestgw.vraa.gov.lv/API-VARAM_VRAA-PaulsTestCalculatorAPI/v1_0/";
        const string identityAddress = "https://viss-portal-test.vraa.gov.lv/IamIdentity.Server/connect/token";

        const string pathnameprivkeyfile = "TestCert.pfx";
        const string passwordprivkey = "123";
        const string scope = "urn:viss.apimanagement-apicall API-SIA.ABC-Viss.ApiManagement.TransactionApi-v1_0-IamStart API-SIA.ABC-Viss.ApiManagement.TransactionApi-v1_0-IamInfo";
        
        const string transactionApiAddress = "https://vissapi-test.vraa.gov.lv/ApiManagement.TransactionApi";
        const string eServiceId = "URN:IVIS:100001:EP-EP186-v1-0";

        const string introspectionAddress = "https://viss-portal-test.vraa.gov.lv/IamIdentity.Server/connect/introspect";
        const string introspectClientId = "urn:viss.apimanagement";
        const string introspectClientSecret = "secret";

        const string jwtClientId = "urn:oauth2:d4ed79fa-8aaf-42aa-86ab-7d3db8defbdc";
        const string referenceClientId = "urn:oauth2:bd9f5fc0-e527-4ab1-8c8a-b870433435f2";

        private readonly HttpClient client;

        // Iespējams pārslēgties starp JWT un Reference tokeniem mainot tokenTypeToUse mainīgā vērtību
        // JWT jādarbojas bez introspekcijas
        // Reference tokeniem ir introspekcija,
        // bet nesūta pieprasījumus uz Multiply vai Divide metodi,
        // jo klients neatbalsta autentifikāciju ar Reference tokeniem
        private readonly TokenType tokenTypeToUse = TokenType.Jwt;

        private readonly TransactionApiClient transactionApiClient;
        private string transactionId;        

        public RestExample()
        {
            var clientCertificate = new X509Certificate2(pathnameprivkeyfile, passwordprivkey);
            string effectiveClientId = tokenTypeToUse == TokenType.Jwt ? jwtClientId : referenceClientId;

            IdentityServerClient identityServerClient = new IdentityServerClient(
                new ClientCredentialsTokenRequest()
                {
                    Address = identityAddress,
                    ClientId = effectiveClientId,
                    Scope = scope,
                    ClientCredentialStyle = ClientCredentialStyle.PostBody,
                    ClientAssertion = new ClientAssertion()
                    {
                        Type = "urn:ietf:params:oauth:client-assertion-type:jwt-bearer",
                        Value = IdentityServerClient.CreateAssertion(
                            clientCertificate,
                            effectiveClientId,
                            identityAddress, 
                            includeCertificate: true, 
                            includeThumbrint: false),
                    }
                },
                new TokenIntrospectionRequest()
                {
                    Address = introspectionAddress,
                    ClientId = introspectClientId,
                    ClientSecret = introspectClientSecret
                }
            );

            var handler = new AuthenticationHandler(identityServerClient, tokenTypeToUse);

            this.transactionApiClient = new TransactionApiClient(new HttpClient(handler)
            {
                BaseAddress = new Uri(transactionApiAddress)
            });

            this.client = new HttpClient(handler)
            {
                BaseAddress = new Uri(baseAddress)
            };
        }

        public async Task RunAsync()
        {
            await MultiplyAsync();
            await DivideAsync();
        }

        private async Task MultiplyAsync()
        {

            var data = new Dictionary<string, object>
                {
                    { "Multiplier", 12 },
                    { "Multiplicand", 12 }
                };

            if (transactionId == null)
            {
                transactionId = await transactionApiClient.StartAsync(eServiceId);
            }

            this.client.DefaultRequestHeaders.Remove("x-transactionId");
            this.client.DefaultRequestHeaders.Add("x-transactionId", transactionId);

            // nesūta pieprasījumu, ja tiek izmantots Reference tokens, jo REST klients to neatbalsta
            if (tokenTypeToUse == TokenType.Jwt)
            {
                using var response = await this.client.PostAsJsonAsync("api/Calculator/Multiply", data);

                if (response.IsSuccessStatusCode)
                {
                    string result = await response.Content.ReadAsStringAsync();
                    Console.WriteLine(result);
                }
                else
                {
                    var errorBody = await response.Content.ReadAsStringAsync();
                    Console.WriteLine("Error response body:");
                    Console.WriteLine(errorBody);
                    throw new Exception($"Failed getting multiply : {response.ReasonPhrase}");
                }
            }
            else
            {
                Console.WriteLine("Multiply operation is only supported with JWT tokens.");
            }
        }

        private async Task DivideAsync()
        {

            var data = new Dictionary<string, object>
            {
                { "Dividend", 10 },
                { "Divisor", 2 }
            };

            if (transactionId == null)
            {
                transactionId = await transactionApiClient.StartAsync(eServiceId);
            }

            this.client.DefaultRequestHeaders.Remove("x-transactionId");
            this.client.DefaultRequestHeaders.Add("x-transactionId", transactionId);
            // nesūta pieprasījumu, ja tiek izmantots Reference tokens, jo REST klients to neatbalsta
            if (tokenTypeToUse == TokenType.Jwt)
            {
                using var response = await this.client.PostAsJsonAsync("api/Calculator/Divide", data);

                if (response.IsSuccessStatusCode)
                {
                    string result = await response.Content.ReadAsStringAsync();
                    Console.WriteLine(result);
                }
                else
                {
                    throw new Exception($"Failed getting divide : {response.ReasonPhrase}");
                }
            }
            else
            {
                Console.WriteLine("Divide operation is only supported with JWT tokens.");
            }
        }
    }
}
