package utils;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

public class IdentityServerClient {
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String CONTENT_TYPE_VALUE = "application/x-www-form-urlencoded";
    private static final String UTF_8 = "UTF-8";
    private static final String ERROR_MESSAGE = "Error occurred while getting TokenMetaData. Status code: ";

    private final PasswordTokenRequest passwordTokenRequest;
    private final ClientCredentialsRequest clientCredentialsRequest;
    private final IntrospectRequest introspectRequest;
    private final Gson gson;

    public IdentityServerClient(PasswordTokenRequest passwordTokenRequest,
                              ClientCredentialsRequest clientCredentialsRequest,
                              IntrospectRequest introspectRequest) {
        this.passwordTokenRequest = passwordTokenRequest;
        this.clientCredentialsRequest = clientCredentialsRequest;
        this.introspectRequest = introspectRequest;
        this.gson = new GsonBuilder().create();
    }

    public Optional<String> getAccessTokenClientCredentials() {
        String requestBody = buildClientCredentialsRequestBody();
        return executeTokenRequest(clientCredentialsRequest.getAddress(), requestBody);
    }

    public Optional<String> getAccessTokenPassword() {
        String requestBody = buildPasswordRequestBody();
        return executeTokenRequest(passwordTokenRequest.getAddress(), requestBody);
    }

    public Optional<IntrospectResponse> getIntrospectionModel(String accessToken) {
        if (accessToken == null || accessToken.isEmpty()) {
            return Optional.empty();
        }

        String requestBody = "token=" + accessToken;
        HttpClient client = new DefaultHttpClient();
        BufferedReader reader = null;

        try {
            HttpPost httpPost = createIntrospectPost(requestBody, false);
            HttpResponse response = client.execute(httpPost);

            if (isSuccessful(response.getStatusLine().getStatusCode())) {
                reader = getReader(response);
                return Optional.of(gson.fromJson(reader, IntrospectResponse.class));
            }
            throw new IOException(ERROR_MESSAGE + response.getStatusLine().getStatusCode());
        } catch (Exception e) {
            System.out.println("Failed to get introspection model: " + e.getMessage());
            return Optional.empty();
        } finally {
            IOUtils.closeQuietly(reader);
            client.getConnectionManager().shutdown();
        }
    }

    public Optional<JSONObject> getIntrospection(String accessToken) {
        if (accessToken == null || accessToken.isEmpty()) {
            return Optional.empty();
        }

        String requestBody = "token=" + accessToken;
        HttpClient client = new DefaultHttpClient();
        BufferedReader reader = null;

        try {
            HttpPost httpPost = createIntrospectPost(requestBody, true);
            HttpResponse response = client.execute(httpPost);

            if (isSuccessful(response.getStatusLine().getStatusCode())) {
                reader = getReader(response);
                return Optional.ofNullable(getParsedObjectByReader(reader));
            }
            throw new IOException(ERROR_MESSAGE + response.getStatusLine().getStatusCode());
        } catch (Exception e) {
            System.out.println("Failed to get introspection: " + e.getMessage());
            return Optional.empty();
        } finally {
            IOUtils.closeQuietly(reader);
            client.getConnectionManager().shutdown();
        }
    }

    private String buildClientCredentialsRequestBody() {
        StringBuilder builder = new StringBuilder()
            .append("grant_type=").append(clientCredentialsRequest.getGrantType())
            .append("&client_assertion_type=").append(encode(clientCredentialsRequest.getClientAssertionType()))
            .append("&client_assertion=").append(encode(clientCredentialsRequest.getClientAssertion()));

        if (clientCredentialsRequest.getScope() != null) {
            builder.append("&scope=").append(encode(clientCredentialsRequest.getScope()));
        }
        return builder.toString();
    }

    private String buildPasswordRequestBody() {
        StringBuilder builder = new StringBuilder()
            .append("grant_type=").append(passwordTokenRequest.getGrantType())
            .append("&username=").append(encode(passwordTokenRequest.getUserName()))
            .append("&password=").append(encode(passwordTokenRequest.getPassword()))
            .append("&client_id=").append(encode(passwordTokenRequest.getClientId()))
            .append("&client_secret=").append(encode(passwordTokenRequest.getClientSecret()));

        if (passwordTokenRequest.getScope() != null) {
            builder.append("&scope=").append(encode(passwordTokenRequest.getScope()));
        }
        return builder.toString();
    }

    private Optional<String> executeTokenRequest(String address, String requestBody) {
        HttpClient client = new DefaultHttpClient();
        BufferedReader reader = null;

        try {
            HttpPost httpPost = new HttpPost(address);
            httpPost.setHeader(CONTENT_TYPE, CONTENT_TYPE_VALUE);
            httpPost.setEntity(new StringEntity(requestBody, UTF_8));

            HttpResponse response = client.execute(httpPost);
            if (isSuccessful(response.getStatusLine().getStatusCode())) {
                reader = getReader(response);
                JSONObject parsedObject = getParsedObjectByReader(reader);
                return Optional.ofNullable(parsedObject)
                        .map(obj -> obj.get("access_token").toString());
            }
            throw new IOException(ERROR_MESSAGE + response.getStatusLine().getStatusCode());
        } catch (Exception e) {
            System.out.println("Failed to get access token: " + e.getMessage());
            return Optional.empty();
        } finally {
            IOUtils.closeQuietly(reader);
            client.getConnectionManager().shutdown();
        }
    }

    private HttpPost createIntrospectPost(String requestBody, boolean useBasicAuth) throws IOException {
        HttpPost httpPost = new HttpPost(introspectRequest.getAddress());
        httpPost.setHeader(CONTENT_TYPE, CONTENT_TYPE_VALUE);

        if (useBasicAuth) {
            String credentials = introspectRequest.getClientId() + ":" + introspectRequest.getClientSecret();
            String encodedAuth = Base64.encodeBase64String(credentials.trim().getBytes(StandardCharsets.UTF_8));
            httpPost.setHeader("Authorization", "Basic " + encodedAuth);
        }

        httpPost.setEntity(new StringEntity(requestBody, UTF_8));
        return httpPost;
    }

    private BufferedReader getReader(HttpResponse response) throws IOException {
        HttpEntity entity = response.getEntity();
        return new BufferedReader(new InputStreamReader(entity.getContent(), UTF_8));
    }

    private boolean isSuccessful(int statusCode) {
        return statusCode == HttpStatus.SC_OK || statusCode == HttpStatus.SC_CREATED;
    }

    private String encode(String value) {
        return URLEncoder.encode(value, StandardCharsets.UTF_8);
    }

    private JSONObject getParsedObjectByReader(BufferedReader reader) throws Exception {
        if (reader == null) return null;
        JSONParser parser = new JSONParser();
        return (JSONObject) parser.parse(reader);
    }
}
